/*-
 * Copyright 2000-2001 John-Mark Gurney.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: cache.c,v 1.5 2003/08/17 13:15:21 jmg Exp $
 *
 */

#include <cachepriv.h>

#include <sys/time.h>
#include <stdlib.h>

/* local prototypes */
static struct cache *init();
static struct cachenode *ninit(cmp_fun);
static int internalcmp(bt_data_t, bt_data_t);
static int gettimestamp();

/* private functions */
static int
gettimestamp()
{
	const int prec = 3;
	struct timeval tv;

	gettimeofday(&tv, NULL);

	return (tv.tv_sec << prec) + (quad_t)tv.tv_usec * (1 << prec) / 1000000;
}

/*
 * XXX - this extra layer of indirection sucks!  If I had perforce, I could
 * import/modify the B-Tree code to do this for me, but I have to make do
 * with CVS. :(
 */
int
internalcmp(void *a, void *b)
{
	struct cachenode *x = a;
	struct cachenode *y = b;

	return x->cn_cmp(x->cn_data, y->cn_data);
}

static struct cache *
init()
{
	struct cache *c;

	if ((c = malloc(sizeof *c)) != NULL) {
		c->c_tree = NULL;
		c->c_heap = NULL;
		c->c_cmp = NULL;
		c->c_num = 0;
	}

	return c;
}

static struct cachenode *
ninit(cmp_fun cmp)
{
	struct cachenode *ret;

	if ((ret = malloc(sizeof *ret)) != NULL) {
		ret->cn_data = NULL;
		ret->cn_hel = NULL;
		ret->cn_cmp = cmp;
	}

	return ret;
}

/* public functions */
struct cache *
lc_create(cmp_fun cmp)
{
	struct cache *c;

	if ((c = init()) != NULL) {
		c->c_tree = bt_create(internalcmp, 512);
		c->c_heap = fh_makekeyheap();
		c->c_cmp = cmp;
	}

	return c;
}

c_data_t
lc_insert(struct cache *c, c_data_t data)
{
	struct cachenode *cn;

	cn = ninit(c->c_cmp);
	cn->cn_data = data;
	cn->cn_hel = fh_insertkey(c->c_heap, gettimestamp(), cn);
	bt_insert(c->c_tree, cn);
	c->c_num++;

	return data;
}

c_data_t
lc_find(struct cache *c, c_data_t obj)
{
	struct cachenode dummy;
	struct cachenode *cn;

	dummy.cn_cmp = c->c_cmp;
	dummy.cn_data = obj;

	if ((cn = bt_find(c->c_tree, &dummy)) == NULL)
		return NULL;

	fh_replacekey(c->c_heap, cn->cn_hel, gettimestamp());

	return cn->cn_data;
}

c_data_t
lc_delmin(struct cache *c)
{
	c_data_t ret;
	struct cachenode *cn;

	cn = fh_extractmin(c->c_heap);
	bt_delete(c->c_tree, cn);
	ret = cn->cn_data;
	c->c_num--;

	free(cn);

	return ret;
}

int
lc_num(struct cache *c)
{
	return c->c_num;
}
